/*
 * bt87x.c - Brooktree Bt878/Bt879 driver for ALSA
 *
 * Copyright (c) Clemens Ladisch <clemens@ladisch.de>
 *
 * based on btaudio.c by Gerd Knorr <kraxel@bytesex.org>
 *
 *
 *  This driver is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This driver is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */





#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/moduleparam.h>
#include <linux/bitops.h>
#include <asm/io.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/control.h>
#include <sound/initval.h>

// ST_070900_ADD
#include "techwell.h"


// ST_070900, Add TW68xx support
#ifdef SNDRV_CARDS
#undef SNDRV_CARDS
#define SNDRV_CARDS 9
#else
#define SNDRV_CARDS 9
#endif

// ST_070900_END



MODULE_AUTHOR("ST Chen, Techwell Inc,");
MODULE_DESCRIPTION("Techwell  TW68xx Audio driver");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("{ { Techwell,TW6805}," "{Techwell, TW68xx}}" ) ;


static int index[SNDRV_CARDS] = {[0 ... (SNDRV_CARDS - 1)] = -2}; /* Exclude the first card */
static char *id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;	/* ID for this card */
static int enable[SNDRV_CARDS] = SNDRV_DEFAULT_ENABLE_PNP;	/* Enable this card */
static int digital_rate[SNDRV_CARDS];	/* digital input rate */
static int load_all;	/* allow to load the non-whitelisted cards */

module_param_array(index, int, NULL, 0444);
MODULE_PARM_DESC(index, "Index value for Bt87x soundcard");
module_param_array(id, charp, NULL, 0444);
MODULE_PARM_DESC(id, "ID string for Bt87x soundcard");
module_param_array(enable, bool, NULL, 0444);
MODULE_PARM_DESC(enable, "Enable Bt87x soundcard");
module_param_array(digital_rate, int, NULL, 0444);
MODULE_PARM_DESC(digital_rate, "Digital input rate for Bt87x soundcard");
module_param(load_all, bool, 0444);
MODULE_PARM_DESC(load_all, "Allow to load the non-whitelisted cards");


static inline u32 snd_bt87x_readl(struct snd_bt87x *chip, u32 reg)
{
	return readl(chip->mmio + reg);
}

static inline void snd_bt87x_writel(struct snd_bt87x *chip, u32 reg, u32 value)
{
	writel(value, chip->mmio + reg);
}

// ST_071000, For RISC
#if 1
static int
snd_bt87x_create_risc
(
	struct snd_bt87x *chip,
	struct snd_pcm_substream *substream,
       	 unsigned int periods,
       	 unsigned int period_bytes
)
{
	unsigned int i, offset;
	u32 *risc;

	printk( "-----> TW68xx: Create RISC-4 ... \n" ) ;

	if (chip->dma_risc.area == NULL) {
		if (snd_dma_alloc_pages(SNDRV_DMA_TYPE_DEV, snd_dma_pci_data(chip->pci),
					PAGE_ALIGN(MAX_RISC_SIZE), &chip->dma_risc) < 0) {
			printk( "<----- TW68xx: Create RISC, ENOMEM Failed ...  \n" ) ;
			return -ENOMEM;
		}
	}

	risc = (u32 *)chip->dma_risc.area ;
	offset = 0 ;

	printk( "periods: %d, period_bytes: %d \n", periods, period_bytes ) ;

	for ( i = 0; i < periods; ++ i ) {
		u32 rest;
		rest = period_bytes ;
		do {
			u32 cmd, len ;
			len = PAGE_SIZE - (offset % PAGE_SIZE);
			if ( len > rest) len = rest ;

			printk( "i: %d, len: %d, offset: %d, rest: %d ... \n", i, len, offset, rest  )  ;

			cmd = (ASTART | len ) ;
			if (len == rest)	cmd |= TRIG_IRQ ;

			*risc++ = cpu_to_le32(cmd) ;
			*risc++ = cpu_to_le32((u32)snd_pcm_sgbuf_get_addr(substream, offset)) ; 	// Get physical address
			offset += len ;
			rest -= len ;

		} while (rest > 0) ;

	}

	*risc++ = cpu_to_le32( JUMP ) ;
	// *risc = cpu_to_le32( chip->dma_risc.addr + ( (*risc+1)*sizeof(risc)*2 )) ;
	// *risc ++ ;
	*risc++ = cpu_to_le32( chip->dma_risc.addr ) ;

	/*
	*risc++ = cpu_to_le32(JUMP) ;
	*risc++ = cpu_to_le32(chip->dma_risc.addr);

	*risc++ = cpu_to_le32(JUMP) ;
	*risc++ = cpu_to_le32(chip->dma_risc.addr);

	*risc++ = cpu_to_le32(JUMP) ;
	*risc++ = cpu_to_le32(chip->dma_risc.addr);
	*/

	chip->line_bytes = period_bytes;
	chip->lines = periods;

	printk( "<----- TW68xx: Create RISC  \n" ) ;

	return 0;

}


#else
static int
snd_bt87x_create_risc
(
	struct snd_bt87x *chip,
	struct snd_pcm_substream *substream,
       	 unsigned int periods,
       	 unsigned int period_bytes
)
{
	struct snd_sg_buf *sgbuf = snd_pcm_substream_sgbuf(substream);
	unsigned int i, offset;
	u32 *risc;

	printk( "-----> TW68xx: Create RISC  \n" ) ;

	if (chip->dma_risc.area == NULL) {
		if (snd_dma_alloc_pages(SNDRV_DMA_TYPE_DEV, snd_dma_pci_data(chip->pci),
					PAGE_ALIGN(MAX_RISC_SIZE), &chip->dma_risc) < 0)
			printk( "<----- TW68xx: Create RISC, ENOMEM Failed ...  \n" ) ;
			return -ENOMEM;
	}

	risc = (u32 *)chip->dma_risc.area ;
	offset = 0;
	*risc++ = cpu_to_le32(RISC_SYNC | RISC_SYNC_FM1);
	*risc++ = cpu_to_le32(0);

	for (i = 0; i < periods; ++i) {
		u32 rest;
		rest = period_bytes;
		do {
			u32 cmd, len ;
			len = PAGE_SIZE - (offset % PAGE_SIZE);
			if (len > rest)
				len = rest;
			cmd = RISC_WRITE | len ;
			if (rest == period_bytes) {
				u32 block = i * 16 / periods;
				cmd |= RISC_SOL;
				cmd |= block << RISC_SET_STATUS_SHIFT;
				cmd |= (~block & 0xf) << RISC_RESET_STATUS_SHIFT;
			}
			if (len == rest)
				cmd |= RISC_EOL | RISC_IRQ;
			*risc++ = cpu_to_le32(cmd);
			*risc++ = cpu_to_le32((u32)snd_pcm_sgbuf_get_addr(sgbuf, offset));
			offset += len;
			rest -= len;
		} while (rest > 0);
	}
	*risc++ = cpu_to_le32(RISC_SYNC | RISC_SYNC_VRO);
	*risc++ = cpu_to_le32(0);
	*risc++ = cpu_to_le32(RISC_JUMP);
	*risc++ = cpu_to_le32(chip->dma_risc.addr);
	chip->line_bytes = period_bytes;
	chip->lines = periods;

	printk( "<----- TW68xx: Create RISC  \n" ) ;

	return 0;

}
#endif
// ST_071000_END


static void snd_bt87x_free_risc(struct snd_bt87x *chip)
{

	printk( "-----> TW68xx: Free RISC  \n" ) ;

	if (chip->dma_risc.area) {
		snd_dma_free_pages(&chip->dma_risc);
		chip->dma_risc.area = NULL;
	}

	printk( "<----- TW68xx: Free RISC  \n" ) ;

}

static void
snd_bt87x_pci_error
(
	struct snd_bt87x *chip,
	unsigned int status
)
{
	u16 pci_status;

	printk( "-----> TW68xx: PCI Error  \n" ) ;

	pci_read_config_word(chip->pci, PCI_STATUS, &pci_status);

	pci_status &= PCI_STATUS_PARITY | PCI_STATUS_SIG_TARGET_ABORT |
		PCI_STATUS_REC_TARGET_ABORT | PCI_STATUS_REC_MASTER_ABORT |
		PCI_STATUS_SIG_SYSTEM_ERROR | PCI_STATUS_DETECTED_PARITY;

	pci_write_config_word(chip->pci, PCI_STATUS, pci_status);

	if (pci_status != PCI_STATUS_DETECTED_PARITY)
		snd_printk( KERN_ERR "Aieee - PCI error! status %#08x, PCI status %#04x\n",  status & ERROR_INTERRUPTS, pci_status);
	else {
		snd_printk(KERN_ERR "Aieee - PCI parity error detected!\n");
		/* error 'handling' similar to aic7xxx_pci.c: */
		chip->pci_parity_errors++;
		if (chip->pci_parity_errors > 20) {
			snd_printk(KERN_ERR "Too many PCI parity errors observed.\n");
			snd_printk(KERN_ERR "Some device on this bus is generating bad parity.\n");
			snd_printk(KERN_ERR "This is an error *observed by*, not *generated by*, this card.\n");
			snd_printk(KERN_ERR "PCI parity error checking has been disabled.\n");
			chip->interrupt_mask &= ~(INT_PPERR | INT_ADMAPERR);
			snd_bt87x_writel(chip, REG_INT_MASK, chip->interrupt_mask);
		}
	}

	printk( "<----- TW68xx: PCI Error  \n" ) ;

}

static irqreturn_t
snd_bt87x_interrupt
(
	int irq,
	void *dev_id
)
{
	struct snd_bt87x *chip = dev_id;
	unsigned int status, irq_status;

	status = snd_bt87x_readl(chip, REG_INT_STAT);

	// printk( "Interrupt:: IRQ_Status: %08x, MASK: %x ... \n", status, chip->interrupt_mask ) ;

	irq_status = status & chip->interrupt_mask;

	if (!irq_status) 	return IRQ_NONE;

	// ST_070900, Add TW68xx support
	// printk( "Interrupt:: IRQ_Status: %08x.....\n", irq_status) ;

	snd_bt87x_writel(chip, REG_INT_STAT, irq_status);

	if ( irq_status & ERROR_INTERRUPTS ) {

		// ST_070900_DEL
		/*
		if (irq_status & (INT_FBUS | INT_FTRGT))
			snd_printk(KERN_WARNING "FIFO overrun, status %#08x\n", status);

		if (irq_status & INT_OCERR)
			snd_printk(KERN_ERR "internal RISC error, status %#08x\n", status);
		*/

		// if ( irq_status & (INT_PPERR | INT_ADMAPERR| INT_PABORT) ) {
		if ( irq_status & (INT_PPERR | INT_PABORT) ) {
			snd_bt87x_pci_error(chip, irq_status) ;
		} else if ( irq_status & INT_ADMAPERR ) {
			// ST_071500, Reset the DMA Engine & FIFO
			snd_bt87x_writel(chip, REG_ADMAC, 0x00 )  ;
			snd_bt87x_writel(chip, REG_ADMAC, 0x03 )  ;
		}

	}


	// ST_0709
	// if ( (irq_status & INT_RISCI) && (chip->reg_control & CTL_ACAP_EN) ) {
	if (  (irq_status & INT_DMAPI ) && ( chip->reg_control & CTL_ACAP_EN ) ) {

		//int current_block, irq_block;

		printk( "Interrupt:: DMAPI !!\n" ) ;

		/* assume that exactly one line has been recorded */
		chip->current_line = (chip->current_line + 1) % chip->lines;

		printk( "Interrupt:: CurrentLind: %d, Lines: %d \n", chip->current_line, chip->lines ) ;

		/* but check if some interrupts have been skipped */
		// ST_070900_DEL
		/*
		current_block = chip->current_line * 16 / chip->lines;

		irq_block = status >> INT_RISCS_SHIFT ;

		if (current_block != irq_block)
			chip->current_line = (irq_block * chip->lines + 15) / 16;
		*/

		snd_pcm_period_elapsed(chip->substream);

	}

	return IRQ_HANDLED;


}


// ST_073000
#if 1

static struct snd_pcm_hardware snd_bt87x_digital_hw = {
	.info = SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_INTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER |
		SNDRV_PCM_INFO_MMAP_VALID,
	.formats = SNDRV_PCM_FMTBIT_S16_LE,
	.rates = SNDRV_PCM_RATE_CONTINUOUS,
	.channels_min = 1,
	.channels_max = 1,
	.buffer_bytes_max = 255 * 4092,
	.period_bytes_min = 1024,
	.period_bytes_max = 4092,
	.periods_min = 2,
	.periods_max = 255,
};


#else

static struct snd_pcm_hardware snd_bt87x_digital_hw = {
	.info = SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_INTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER |
		SNDRV_PCM_INFO_MMAP_VALID,
	.formats = SNDRV_PCM_FMTBIT_S16_LE,
	.rates = 0, /* set at runtime */
	.channels_min = 2,
	.channels_max = 2,
	.buffer_bytes_max = 255 * 4092,
	.period_bytes_min = 32,
	.period_bytes_max = 4092,
	.periods_min = 2,
	.periods_max = 255,
};

#endif


// ST_071500
#if 1
static struct snd_pcm_hardware snd_bt87x_analog_hw = {
	.info = 		SNDRV_PCM_INFO_MMAP |
				SNDRV_PCM_INFO_INTERLEAVED |
				SNDRV_PCM_INFO_BLOCK_TRANSFER |
				SNDRV_PCM_INFO_MMAP_VALID,
	.formats =		SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S8,
	.rates = 		SNDRV_PCM_RATE_KNOT,
	.rate_min 			= 8000,
	.rate_max 		= 8000,
	.channels_min 		= 1,
	.channels_max 		= 1,
	.buffer_bytes_max 	= 255 * 4092,
	.period_bytes_min 	= 1024,
	.period_bytes_max 	= 4092,
	.periods_min 		= 1,
	.periods_max 		= 1024,
};

#else
static struct snd_pcm_hardware snd_bt87x_analog_hw = {
	.info = 		SNDRV_PCM_INFO_MMAP |
				SNDRV_PCM_INFO_INTERLEAVED |
				SNDRV_PCM_INFO_BLOCK_TRANSFER |
				SNDRV_PCM_INFO_MMAP_VALID,
	.formats =		SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S8,
	.rates = 		SNDRV_PCM_RATE_KNOT,
	.rate_min = ANALOG_CLOCK / CLOCK_DIV_MAX,
	.rate_max = ANALOG_CLOCK / CLOCK_DIV_MIN,
	.channels_min = 1,
	.channels_max = 1,
	.buffer_bytes_max = 255 * 4092,
	.period_bytes_min = 32,
	.period_bytes_max = 4092,
	.periods_min = 2,
	.periods_max = 255,
};
#endif

// ST_070900, Add TW68xx support
#if 1
// Copy this part from BT87x.c_org
static int
snd_bt87x_set_digital_hw
(
	struct snd_bt87x *chip,
	struct snd_pcm_runtime *runtime
)
{

// ST_070900, Add Tw68xx support
#if 1
//	static struct {
//		int rate;
//		unsigned int bit;
//	} ratebits[] = {
//		{8000, SNDRV_PCM_RATE_8000}
//	};
#else
	static struct {
		int rate;
		unsigned int bit;
	} ratebits[] = {
		{8000, SNDRV_PCM_RATE_8000},
		{11025, SNDRV_PCM_RATE_11025},
		{16000, SNDRV_PCM_RATE_16000},
		{22050, SNDRV_PCM_RATE_22050},
		{32000, SNDRV_PCM_RATE_32000},
		{44100, SNDRV_PCM_RATE_44100},
		{48000, SNDRV_PCM_RATE_48000}
	};
#endif

//	int i;

	printk( "-----> TW68xx: Set digital hw  \n" ) ;

	chip->reg_control |= CTL_DA_IOM_DA;
	runtime->hw = snd_bt87x_digital_hw;
	runtime->hw.rates = SNDRV_PCM_RATE_KNOT;

	// ST_073000
	/*
	for (i = 0; i < ARRAY_SIZE(ratebits); i ++ ) {

		printk( "dig_rate: %d, rate[%d]: %d	...\n", chip->dig_rate, i,   ratebits[i].rate ) ;

		if (chip->dig_rate == ratebits[i].rate) {

			printk( "==> ratenbits: %x ...\n", ratebits[i].bit ) ;

			runtime->hw.rates = ratebits[i].bit;
			break;
		}
	}
	*/


	runtime->hw.rate_min = chip->dig_rate;
	runtime->hw.rate_max = chip->dig_rate;

	printk( "<----- TW68xx: Set digital hw  \n" ) ;

	return 0;

}



#else
static int snd_bt87x_set_digital_hw(struct snd_bt87x *chip, struct snd_pcm_runtime *runtime)
{
	chip->reg_control |= CTL_DA_IOM_DA | CTL_A_PWRDN;
	runtime->hw = snd_bt87x_digital_hw;
	runtime->hw.rates = snd_pcm_rate_to_rate_bit(chip->board.dig_rate);
	runtime->hw.rate_min = chip->board.dig_rate;
	runtime->hw.rate_max = chip->board.dig_rate;
	return 0;
}
#endif


static int snd_bt87x_set_analog_hw(struct snd_bt87x *chip, struct snd_pcm_runtime *runtime)
{
	static struct snd_ratnum analog_clock = {
		.num = ANALOG_CLOCK,
		.den_min = CLOCK_DIV_MIN,
		.den_max = CLOCK_DIV_MAX,
		.den_step = 1
	};
	static struct snd_pcm_hw_constraint_ratnums constraint_rates = {
		.nrats = 1,
		.rats = &analog_clock
	};

	printk( "-----> TW68xx: Set analog hw  \n" ) ;

	chip->reg_control &= ~(CTL_DA_IOM_DA | CTL_A_PWRDN);
	runtime->hw = snd_bt87x_analog_hw;

	printk( "<----- TW68xx: Set analog hw  \n" ) ;

	return snd_pcm_hw_constraint_ratnums(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					     &constraint_rates);
}

static int
snd_bt87x_pcm_open
(
	struct snd_pcm_substream *substream
)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;
	int err;

	printk( "-----> TW68xx:  PCM open \n" ) ;

	if (test_and_set_bit(0, &chip->opened)){
		printk( "<----- TW68xx:  PCM open with EBUSY ... \n" ) ;
		return -EBUSY;
	}

	if (substream->pcm->device == DEVICE_DIGITAL) {
		err = snd_bt87x_set_digital_hw(chip, runtime);
	} else {
		err = snd_bt87x_set_analog_hw(chip, runtime);
	}

	if (err < 0) goto _error;

	err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS);

	if (err < 0) goto _error;

	chip->substream = substream;

	printk( "<----- TW68xx:  PCM open success ...\n" ) ;

	return 0;

_error:

	clear_bit(0, &chip->opened);

	smp_mb__after_clear_bit();

	printk( "-----> TW68xx:  PCM open with ERROR ...\n" ) ;

	return err;
}

static int
snd_bt87x_close
(
	struct snd_pcm_substream *substream
)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);

	printk( " -----> snd_bt87x_close \n" ) ;

	spin_lock_irq(&chip->reg_lock);
	chip->reg_control |= CTL_A_PWRDN;

	// ST_070900, Add TW68xx support
	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
	snd_bt87x_writel( chip, REG_AANACTL, chip->reg_control ) ;

	spin_unlock_irq(&chip->reg_lock);

	printk( "< ----- snd_bt87x_close \n" ) ;

	chip->substream = NULL;
	clear_bit(0, &chip->opened);
	smp_mb__after_clear_bit();
	return 0;
}

static int
snd_bt87x_hw_params
(
	struct snd_pcm_substream *substream,
	struct snd_pcm_hw_params *hw_params
)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);
	int err;

	printk( "-----> TW68xx:  HW Params  \n" ) ;

	err = snd_pcm_lib_malloc_pages( substream, params_buffer_bytes(hw_params) ) ;

	if (err < 0) {
		printk( "<----- TW68xx:  HW Params with ERROR...	\n" ) ;
		return err;
	}

	printk( "<----- TW68xx:  HW Params: Periods: %d, PeriodBytes: %d  ...  \n",
		params_periods(hw_params),  params_period_bytes(hw_params) ) ;

	return snd_bt87x_create_risc(chip, substream,
				     params_periods(hw_params),
				     params_period_bytes(hw_params));

}

static int snd_bt87x_hw_free(struct snd_pcm_substream *substream)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);

	snd_bt87x_free_risc(chip);
	snd_pcm_lib_free_pages(substream);
	return 0;
}

static int
snd_bt87x_prepare
(
	struct snd_pcm_substream *substream
)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;
	int decimation;

	printk( "----->snd_bt87x_prepare\n" ) ;

	spin_lock_irq(&chip->reg_lock);

// ST_070900, Add TW68xx support
#if 1
	// ST_071100, I try to keep the "reg_control" value here
	chip->reg_control &= ~(CTL_DA_SDR_MASK | CTL_DA_SBR);

	decimation = (ANALOG_CLOCK + runtime->rate / 4) / runtime->rate;

	chip->reg_control |= decimation << CTL_DA_SDR_SHIFT;

	if (runtime->format == SNDRV_PCM_FORMAT_S8) chip->reg_control |= CTL_DA_SBR;

	// ST_071500, Channel Select
	/* ST_0127 */
	// snd_bt87x_writel(chip, REG_AANACTL, 0x1080A);					// 0x048
	snd_bt87x_writel(chip, REG_AANACTL, 0x10820);					// 0x048

	// Sample Rate setting
	snd_bt87x_writel(chip, REG_PCI_AVIDCLK_REF, (0x0009B383*2) ) ;	// 27 MHz / 8000
	//snd_bt87x_writel(chip, REG_PCI_AVIDCLK_REF, 0x00136B07 ) ;	// 27 MHz / 8000

	// snd_bt87x_writel(chip, REG_PCI_CLK_REF, 0x0003F8CD ) ;			// 66 MHz / 8000
	//snd_bt87x_writel(chip, REG_PCI_CLK_REF, 0x0007F19A ) ;			// 33 MHz / 8000
	snd_bt87x_writel(chip, REG_PCI_CLK_REF, 0x000FE334 ) ;

	// Other
	snd_bt87x_writel(chip, REG_ACTL1, 0x00280000 ) ;
	snd_bt87x_writel(chip, REG_APACKET, 0x000000 ) ;
	snd_bt87x_writel(chip, REG_ACORE, 0x000000 ) ;

	// ST_080400, For 16bits Mono
	//snd_bt87x_writel(chip, REG_ACTL2, 0x0101 ) ;					//0x018
	snd_bt87x_writel(chip, REG_ACTL2, 0x111 ) ;


//	snd_bt87x_writel(chip, REG_ACLK_SEL, 0x00048 ) ;				// 0x040, PCI clock

    snd_bt87x_writel(chip, REG_ACLK_SEL, 0x00049 ) ;				// 0x040, video clock //denio

	//


#else
	chip->reg_control &= ~(CTL_DA_SDR_MASK | CTL_DA_SBR);
	decimation = (ANALOG_CLOCK + runtime->rate / 4) / runtime->rate;
	chip->reg_control |= decimation << CTL_DA_SDR_SHIFT;
	if (runtime->format == SNDRV_PCM_FORMAT_S8)
		chip->reg_control |= CTL_DA_SBR;
	snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
#endif

	spin_unlock_irq(&chip->reg_lock);

	printk( "<-----snd_bt87x_prepare\n" ) ;

	return 0;
}



static int
snd_bt87x_start
(
	struct snd_bt87x *chip
)
{

	printk( "----->snd_bt87x_start\n" ) ;

	spin_lock(&chip->reg_lock);

// ST_070900, Add TW68xx support
#if 1
	chip->current_line = 0;
	chip->reg_control |= CTL_FIFO_ENABLE | CTL_RISC_ENABLE | CTL_ACAP_EN;

	// ST_071500, move this to "Prepare call back"
	/*
	snd_bt87x_writel(chip, REG_ACTL1, 0x00280000 ) ;
	snd_bt87x_writel(chip, REG_APACKET, 0x000000 ) ;
	snd_bt87x_writel(chip, REG_ACORE, 0x000000 ) ;
	snd_bt87x_writel(chip, REG_AANACTL, 0x010810 ) ;
	snd_bt87x_writel(chip, REG_ACTL2, 0x0101 ) ;
	snd_bt87x_writel(chip, REG_ACLK_SEL, 0x00048 ) ;
	snd_bt87x_writel(chip, REG_PCI_CLK_REF, 0x0003F8CD ) ;
	snd_bt87x_writel(chip, REG_PCI_AVIDCLK_REF, 0x0009B383 ) ;
	*/

	chip->interrupt_mask |= ( INT_OFLOW |INT_PABORT | INT_PPERR | INT_DMAPI | INT_ADMAPERR ) ;
	snd_bt87x_writel(chip, REG_INT_MASK, chip->interrupt_mask);

	//printk( "Start the RISC program, RISC Add: %x, INT_MAK: %x ...\n",
	//		chip->dma_risc.addr, snd_bt87x_readl(chip, REG_INT_MASK)  ) ;

	snd_bt87x_writel(chip, REG_ADMAC, 0x00 )  ;

	snd_bt87x_writel(chip, REG_RISC_STRT_ADD, chip->dma_risc.addr);

	snd_bt87x_writel(chip, REG_ADMAC, ( CTL_FIFO_ENABLE|CTL_RISC_ENABLE ) ) ;

	//printk( "ADMAC: %x \n", snd_bt87x_readl( chip, REG_ADMAC ) ) ;

	// snd_bt87x_writel(chip, REG_PACKET_LEN, chip->line_bytes | (chip->lines << 16));
	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);

#else
	chip->current_line = 0;
	chip->reg_control |= CTL_FIFO_ENABLE | CTL_RISC_ENABLE | CTL_ACAP_EN;
	snd_bt87x_writel(chip, REG_RISC_STRT_ADD, chip->dma_risc.addr);
	snd_bt87x_writel(chip, REG_PACKET_LEN,
			 chip->line_bytes | (chip->lines << 16));
	snd_bt87x_writel(chip, REG_INT_MASK, chip->interrupt_mask);
	snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
#endif
	spin_unlock(&chip->reg_lock);

	printk( "<-----snd_bt87x_start\n" ) ;

	return 0;

}

static int snd_bt87x_stop(struct snd_bt87x *chip)
{
	spin_lock(&chip->reg_lock);

	printk("----->snd_bt87x_stop\n") ;

	// ST_070900, Add TW68xx Support
	chip->reg_control &= ~(CTL_FIFO_ENABLE | CTL_RISC_ENABLE | CTL_ACAP_EN);
	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
	snd_bt87x_writel(chip, REG_ADMAC, 0x00 ) ;
	// ST_END

	snd_bt87x_writel(chip, REG_INT_MASK, 0);
	snd_bt87x_writel(chip, REG_INT_STAT, MY_INTERRUPTS);

	printk("<-----snd_bt87x_stop\n") ;

	spin_unlock(&chip->reg_lock);

	return 0;

}

static int
snd_bt87x_trigger
(
	struct snd_pcm_substream *substream,
	int cmd
)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);

	printk( "----->snd_bt87x_trigger\n" ) ;

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		return snd_bt87x_start(chip);
	case SNDRV_PCM_TRIGGER_STOP:
		return snd_bt87x_stop(chip);
	default:
		return -EINVAL;
	}
}

static snd_pcm_uframes_t snd_bt87x_pointer(struct snd_pcm_substream *substream)
{
	struct snd_bt87x *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;

	printk( "----->snd_bt87x_pointer: currentLine: %d, LineBytes: %d \n",chip->current_line, chip->line_bytes  ) ;

	return (snd_pcm_uframes_t)bytes_to_frames(runtime, chip->current_line * chip->line_bytes);
}

static struct snd_pcm_ops snd_bt87x_pcm_ops = {
	.open = snd_bt87x_pcm_open,
	.close = snd_bt87x_close,
	.ioctl = snd_pcm_lib_ioctl,
	.hw_params = snd_bt87x_hw_params,
	.hw_free = snd_bt87x_hw_free,
	.prepare = snd_bt87x_prepare,
	.trigger = snd_bt87x_trigger,
	.pointer = snd_bt87x_pointer,
	.page = snd_pcm_sgbuf_ops_page,
};

static int snd_bt87x_capture_volume_info(struct snd_kcontrol *kcontrol,
					 struct snd_ctl_elem_info *info)
{
	info->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	info->count = 1;
	info->value.integer.min = 0;

	// ST_080300, For Volume control
	// info->value.integer.max = 15;
	info->value.integer.max = 0x3F ;

	return 0;
}

static int snd_bt87x_capture_volume_get(struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *value)
{
	struct snd_bt87x *chip = snd_kcontrol_chip(kcontrol);


	// ST_080300, For volume control
	chip->m_Volume = ( snd_bt87x_readl( chip, REG_AANACTL ) & REG_VOLUME_MASK ) ;
	 value->value.integer.value[0] = chip->m_Volume ;
	 value->value.integer.value[1] = chip->m_Volume ;
	// value->value.integer.value[0] = (chip->reg_control & CTL_A_GAIN_MASK) >> CTL_A_GAIN_SHIFT;

	return 0;
}

static int snd_bt87x_capture_volume_put(struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *value)
{
	struct snd_bt87x *chip = snd_kcontrol_chip(kcontrol);
	u32 old_control;
	int changed;
	u32	TempVolume ;

	// printk ( "----->TW68xx: Capture Volume Put \n" ) ;

	spin_lock_irq(&chip->reg_lock);

// ST_070900_DEL
#if 1

	old_control = chip->reg_control;

	chip->reg_control = (chip->reg_control & ~CTL_A_GAIN_MASK)
		| (value->value.integer.value[0] << CTL_A_GAIN_SHIFT);

	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);

	changed = old_control != chip->reg_control;

	// ST_071500, For volume control, adjust the ADC Gain ...
	TempVolume = snd_bt87x_readl( chip, REG_AANACTL ) & (~REG_VOLUME_MASK ) ;

	//printk ( "----->TW68xx: snd_bt87x_capture_volume_put: Value: %d  \n", value->value.integer.value[0] ) ;

	TempVolume |= ( value->value.integer.value[0] & REG_VOLUME_MASK ) ;

	snd_bt87x_writel(chip, REG_AANACTL,  TempVolume ) ;

#else
	old_control = chip->reg_control;
	chip->reg_control = (chip->reg_control & ~CTL_A_GAIN_MASK)
		| (value->value.integer.value[0] << CTL_A_GAIN_SHIFT);
	snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
	changed = old_control != chip->reg_control;
#endif
	spin_unlock_irq(&chip->reg_lock);

	// printk ( "<-----TW68xx: Capture Volume Put \n" ) ;

	return changed;

}

static struct snd_kcontrol_new snd_bt87x_capture_volume = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "Capture Volume",
	.info = snd_bt87x_capture_volume_info,
	.get = snd_bt87x_capture_volume_get,
	.put = snd_bt87x_capture_volume_put,
};

#if 0

// ST_070900, Add Tw68xx support
#if 1
// Copy this part from bt87x.c_org
static int
snd_bt87x_capture_boost_info
(
	struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_info *info
)
{

	printk( "-----> TW68xx: Capture boost info  \n" ) ;

	info->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	info->count = 1;
	info->value.integer.min = 0;
	info->value.integer.max = 1;

	printk( "<----- TW68xx: Capture boost info  \n" ) ;

	return 0;
}

#else
#define snd_bt87x_capture_boost_info	snd_ctl_boolean_mono_info
#endif

static int snd_bt87x_capture_boost_get(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *value)
{
	struct snd_bt87x *chip = snd_kcontrol_chip(kcontrol);

	printk( "-----> TW68xx: Capture boost get  \n" ) ;

	// ST_080300, For boost mode control
	// value->value.integer.value[0] = !! (chip->reg_control & CTL_A_G2X);
	value->value.integer.value[0] = chip->m_BoostMode ;	// 0: Disable, 1: Enable
	//

	printk( "<----- TW68xx: Capture boost get  \n" ) ;

	return 0;
}

static int snd_bt87x_capture_boost_put(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *value)
{
	struct snd_bt87x *chip = snd_kcontrol_chip(kcontrol);
	//u32 old_control;
	int changed;

	printk( "-----> TW68xx: Capture boost put   \n" ) ;

	spin_lock_irq(&chip->reg_lock);

// ST_070900_DEL
#if 1
	// ST_080300, For boost control
	/*
	old_control = chip->reg_control;
	chip->reg_control = (chip->reg_control & ~CTL_A_G2X)
		| (value->value.integer.value[0] ? CTL_A_G2X : 0);
	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
	changed = chip->reg_control != old_control;
	*/

#else
	old_control = chip->reg_control;
	chip->reg_control = (chip->reg_control & ~CTL_A_G2X)
		| (value->value.integer.value[0] ? CTL_A_G2X : 0);
	snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
	changed = chip->reg_control != old_control;

#endif

	spin_unlock_irq(&chip->reg_lock);

	printk( "<----- TW68xx: Capture boost put   \n" ) ;

	return changed ;

}


static struct snd_kcontrol_new snd_bt87x_capture_boost = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "Capture Boost",
	.info = snd_bt87x_capture_boost_info,
	.get = snd_bt87x_capture_boost_get,
	.put = snd_bt87x_capture_boost_put,
};

#endif


// ST_071500
#if 1
static int
snd_bt87x_capture_source_info
(
	struct snd_kcontrol *kcontrol,
	 struct snd_ctl_elem_info *info
)
{
	static char *texts[1] = { "Capture" } ;

	//printk( "-----> TW68xx: Capture source info  \n" ) ;

	info->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	info->count = 1;
	info->value.enumerated.items = 1;
	if (info->value.enumerated.item > 2)
		info->value.enumerated.item = 2;
	strcpy(info->value.enumerated.name, texts[info->value.enumerated.item]);

	//printk( "<----- TW68xx: Capture source info  \n" ) ;

	return 0 ;

}

#else
static int snd_bt87x_capture_source_info(struct snd_kcontrol *kcontrol,
					 struct snd_ctl_elem_info *info)
{
	static char *texts[3] = {"TV Tuner", "FM", "Mic/Line"};

	//printk( "-----> TW68xx: Capture source info  \n" ) ;

	info->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	info->count = 1;
	info->value.enumerated.items = 3;
	if (info->value.enumerated.item > 2)
		info->value.enumerated.item = 2;
	strcpy(info->value.enumerated.name, texts[info->value.enumerated.item]);

	//printk( "<----- TW68xx: Capture source info  \n" ) ;

	return 0;
}
#endif


static int
snd_bt87x_capture_source_get
(
	struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *value
)
{
	//struct snd_bt87x *chip = snd_kcontrol_chip(kcontrol);

	// printk( "-----> TW68xx: Capture source get  \n" ) ;

	// ST_071500, Set to Channel # 1 ( Total; 1~4 )
	// value->value.enumerated.item[0] = (chip->reg_control & CTL_A_SEL_MASK) >> CTL_A_SEL_SHIFT;
	value->value.enumerated.item[0] = 0x00 ;

	// printk( "<----- TW68xx: Capture source get \n" ) ;

	return 0;
}

static int
snd_bt87x_capture_source_put
(
	struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *value
)
{
	struct snd_bt87x *chip = snd_kcontrol_chip(kcontrol);
	u32 old_control;
	int changed;

	spin_lock_irq(&chip->reg_lock);

	// printk( "----->TW68xx: Capture Source Put \n" ) ;

// ST_070900_DEL
#if 1

	old_control = chip->reg_control;

	chip->reg_control = (chip->reg_control & ~CTL_A_SEL_MASK)
		| (value->value.enumerated.item[0] << CTL_A_SEL_SHIFT);

	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);

	// ST_071500, Set channel to # 1 ( Total: 1~4 )
	// changed = chip->reg_control != old_control ;
	changed = 1  ;

#else
	old_control = chip->reg_control;
	chip->reg_control = (chip->reg_control & ~CTL_A_SEL_MASK)
		| (value->value.enumerated.item[0] << CTL_A_SEL_SHIFT);
	snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);
	changed = chip->reg_control != old_control;
#endif

	spin_unlock_irq(&chip->reg_lock);

	// printk( "<-----TW68xx: Capture Source Put \n" ) ;

	return changed;

}

static struct snd_kcontrol_new snd_bt87x_capture_source = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "Capture Source",
	.info = snd_bt87x_capture_source_info,
	.get = snd_bt87x_capture_source_get,
	.put = snd_bt87x_capture_source_put,
};

static int snd_bt87x_free(struct snd_bt87x *chip)
{

	printk( "-----> TW68xx: Free \n" ) ;

	/* ST_0817 */
	/*
	if (chip->mmio) {

		snd_bt87x_stop(chip);

		if (chip->irq >= 0)
			synchronize_irq(chip->irq);
		iounmap(chip->mmio);
	}

	if (chip->irq >= 0)
		free_irq(chip->irq, chip);
	*/
	if (chip->mmio) {
		snd_bt87x_stop(chip);
	}

	if (chip->irq >= 0)
		free_irq(chip->irq, chip);

	if ( chip->mmio )
		pci_iounmap( chip->pci, chip->mmio ) ;


	pci_release_regions(chip->pci);

	pci_disable_device(chip->pci);

	kfree(chip);

	printk( "<----- TW68xx: Free \n" ) ;

	return 0;
}


static int
snd_bt87x_dev_free
(
	struct snd_device *device
)
{
	struct snd_bt87x *chip = device->device_data;
	return snd_bt87x_free(chip);
}



static int __devinit
snd_bt87x_pcm
(
	struct snd_bt87x *chip,
	int device,
	char *name
)
{
	int err;
	struct snd_pcm *pcm;

	printk( "-----> snd_bt87x_pcm \n" ) ;

	err = snd_pcm_new(chip->card, name, device, 0, 1, &pcm);

	if (err < 0) return err;

	pcm->private_data = chip;
	strcpy(pcm->name, name);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_bt87x_pcm_ops);

	printk( "<----- snd_bt87x_pcm \n" ) ;

	// ST_071500
#if 1
	snd_pcm_lib_preallocate_pages_for_all(
				pcm,
				SNDRV_DMA_TYPE_DEV_SG,
				snd_dma_pci_data(chip->pci),
				128 * 1024,
				ALIGN(255 * 4092, 1024)) ;
	return 0 ;

#else
	return snd_pcm_lib_preallocate_pages_for_all(pcm,
						     	SNDRV_DMA_TYPE_DEV_SG,
						     	snd_dma_pci_data(chip->pci),
							128 * 1024,
							ALIGN(255 * 4092, 1024));

#endif
}



static int __devinit
snd_bt87x_create
(
	struct snd_card *card,
	struct pci_dev *pci,
	struct snd_bt87x **rchip
)
{
	struct snd_bt87x *chip;
	int err;
	static struct snd_device_ops ops = {
		.dev_free = snd_bt87x_dev_free
	} ;

	printk( "-----> TW68xx: Create \n" ) ;

	*rchip = NULL;

	err = pci_enable_device(pci);

	if (err < 0)
		return err;

	chip = kzalloc(sizeof(*chip), GFP_KERNEL);

	if (!chip) {
		pci_disable_device(pci);
		return -ENOMEM;
	}

	chip->card = card;
	chip->pci = pci;
	chip->irq = -1;
	spin_lock_init(&chip->reg_lock);

	// ST_070900, Add TW68xx Support
	// if ((err = pci_request_regions(pci, "Bt87x audio")) < 0) {
	if ((err = pci_request_regions(pci, "TW6805 Audio")) < 0) {
		kfree(chip);
		pci_disable_device(pci);
		return err;
	}

	/* ST_0817*/
	// chip->mmio = ioremap_nocache(pci_resource_start(pci, 0),  pci_resource_len(pci, 0)) ;
	chip->mmio = pci_iomap( pci, 0,  pci_resource_len(pci, 0) ) ;


	if (!chip->mmio) {
		snd_printk(KERN_ERR "cannot remap io memory\n");
		err = -ENOMEM;
		goto fail;
	}

	// ST_071400, Reset the Audio Device
	snd_bt87x_writel( chip, REG_INT_ARSTN, 0x00  ) ;
	snd_bt87x_writel( chip, REG_INT_ARSTN, 0x01  ) ;
	//

	// ST_070900, Add TW68xx Support
	chip->reg_control = CTL_A_PWRDN | CTL_DA_ES2 | CTL_PKTP_16 | (15 << CTL_DA_SDR_SHIFT);
	// snd_bt87x_writel(chip, REG_GPIO_DMA_CTL, chip->reg_control);

	// chip->reg_control =  0x01080A ;
	snd_bt87x_writel( chip, REG_AANACTL, 0x01080A  ) ;

	// ST_080300, For boost mode control
	chip->m_BoostMode = 0 ;		// 0: Disable, 1: Enable

	chip->interrupt_mask = MY_INTERRUPTS ;

	snd_bt87x_writel(chip, REG_INT_MASK, 0) ;

	snd_bt87x_writel( chip, REG_INT_STAT, MY_INTERRUPTS ) ;

	// ST_070900. Add Tw68xx support
	// err = request_irq(pci->irq, snd_bt87x_interrupt, IRQF_SHARED, "Bt87x audio", chip);
	err = request_irq(pci->irq, snd_bt87x_interrupt, IRQF_SHARED, "TW6805 Audio", chip);

	if (err < 0) {
		printk(KERN_ERR "cannot grab irq %d\n", pci->irq);
		goto fail;
	}

	chip->irq = pci->irq;

	pci_set_master(pci);

	synchronize_irq(chip->irq) ;

	err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, chip, &ops);

	if (err < 0) goto fail ;

	snd_card_set_dev(card, &pci->dev);

	*rchip = chip;

	printk( "<----- TW68xx: Create success ... \n" ) ;

	return 0;

fail:

	snd_bt87x_free(chip);

	printk( "<----- TW68xx: Create Failed ... \n" ) ;

	return err;


}


#define BT_DEVICE(chip, subvend, subdev, id) \
	{ .vendor = PCI_VENDOR_ID_TECHWELL, \
	  .device = chip, \
	  .subvendor = subvend, .subdevice = subdev, \
	  .driver_data = SND_BT87X_BOARD_ ## id }
/* driver_data is the card id for that device */

static struct pci_device_id snd_bt87x_ids[] = {

	// ST_070900, Add TW68xx support
	BT_DEVICE(PCI_DEVICE_ID_TW6805, PCI_ANY_ID, PCI_ANY_ID, TECHWELL),
	// ST_END

	/* Denio 20100120 Add TW6816 support */
	BT_DEVICE(PCI_DEVICE_ID_TW6816_14, PCI_ANY_ID, PCI_ANY_ID, TECHWELL),
	BT_DEVICE(PCI_DEVICE_ID_TW6816_15, PCI_ANY_ID, PCI_ANY_ID, TECHWELL),
	BT_DEVICE(PCI_DEVICE_ID_TW6816_16, PCI_ANY_ID, PCI_ANY_ID, TECHWELL),
    BT_DEVICE(PCI_DEVICE_ID_TW6816_17, PCI_ANY_ID, PCI_ANY_ID, TECHWELL),

	/* Hauppauge WinTV series */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x0070, 0x13eb, GENERIC),
	/* Hauppauge WinTV series */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_879, 0x0070, 0x13eb, GENERIC),
	/* Viewcast Osprey 200 */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x0070, 0xff01, OSPREY2x0),
	/* Viewcast Osprey 440 (rate is configurable via gpio) */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x0070, 0xff07, OSPREY440),
	/* ATI TV-Wonder */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x1002, 0x0001, GENERIC),
	/* Leadtek Winfast tv 2000xp delux */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x107d, 0x6606, GENERIC),
	/* Voodoo TV 200 */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x121a, 0x3000, GENERIC),
	/* AVerMedia Studio No. 103, 203, ...? */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x1461, 0x0003, AVPHONE98),
	/* Prolink PixelView PV-M4900 */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0x1554, 0x4011, GENERIC),
	/* Pinnacle  Studio PCTV rave */
	BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, 0xbd11, 0x1200, GENERIC),
	{ }
};
MODULE_DEVICE_TABLE(pci, snd_bt87x_ids);

/* cards known not to have audio
 * (DVB cards use the audio function to transfer MPEG data) */
static struct {
	unsigned short subvendor, subdevice;
} blacklist[] __devinitdata = {
	{0x0071, 0x0101}, /* Nebula Electronics DigiTV */
	{0x11bd, 0x001c}, /* Pinnacle PCTV Sat */
	{0x11bd, 0x0026}, /* Pinnacle PCTV SAT CI */
	{0x1461, 0x0761}, /* AVermedia AverTV DVB-T */
	{0x1461, 0x0771}, /* AVermedia DVB-T 771 */
	{0x1822, 0x0001}, /* Twinhan VisionPlus DVB-T */
	{0x18ac, 0xd500}, /* DVICO FusionHDTV 5 Lite */
	{0x18ac, 0xdb10}, /* DVICO FusionHDTV DVB-T Lite */
	{0x18ac, 0xdb11}, /* Ultraview DVB-T Lite */
	{0x270f, 0xfc00}, /* Chaintech Digitop DST-1000 DVB-S */
	{0x7063, 0x2000}, /* pcHDTV HD-2000 TV */
	{0x0568, 0x9717}, // ST_070900, Add TW68xx support
};

static struct pci_driver driver;

/* return the id of the card, or a negative value if it's blacklisted */
static int __devinit snd_bt87x_detect_card(struct pci_dev *pci)
{
	int i;
	const struct pci_device_id *supported;

	printk( "-----> TW68xx: Detect Card  \n" ) ;

	supported = pci_match_id(snd_bt87x_ids, pci);

	if (supported && supported->driver_data > 0) {
		return supported->driver_data;
	}

	for (i = 0; i < ARRAY_SIZE(blacklist); ++i)
		if (blacklist[i].subvendor == pci->subsystem_vendor &&
		    blacklist[i].subdevice == pci->subsystem_device) {
			snd_printdd(KERN_INFO "card %#04x-%#04x:%#04x has no audio\n",
				    pci->device, pci->subsystem_vendor, pci->subsystem_device);
			return -EBUSY;
		}

	snd_printk(KERN_INFO "unknown card %#04x-%#04x:%#04x\n",
		   pci->device, pci->subsystem_vendor, pci->subsystem_device);\

	snd_printk(KERN_DEBUG "please mail id, board name, and, "
		   "if it works, the correct digital_rate option to "
		   "<alsa-devel@alsa-project.org>\n");

	printk( "<----- TW68xx: Detect card, Success with UNKNOWN ... \n" ) ;

	return SND_BT87X_BOARD_UNKNOWN;

}



// ST_071400
#if 1
static int
__devinit snd_bt87x_probe
(
	struct pci_dev *pci,
     	const struct pci_device_id *pci_id
)
{
	static int dev;
	struct snd_card *card;
	struct snd_bt87x *chip;
	int err, rate;

	rate = pci_id->driver_data;

	printk( "----->snd_bt87x_probe(rate: %d) \n", rate  ) ;

	rate = 8000 ;

	if (! rate) {
		if ((rate = snd_bt87x_detect_card(pci)) <= 0) {
			printk( "<-----snd_bt87x_probe; detect card erro ....\n" ) ;
			return -ENODEV;
		}
	}

	if (dev >= SNDRV_CARDS) {
		printk( "<-----snd_bt87x_probe: failed - 1 ....\n" ) ;
		return -ENODEV;
	}

	if (!enable[dev]) {
		++dev;
		printk( "<-----snd_bt87x_probe: failed -2 ...\n" ) ;
		return -ENOENT;
	}

	 err = snd_card_create(index[dev], id[dev], THIS_MODULE, 0, &card);
         if (err < 0)
                 return err;


	err = snd_bt87x_create(card, pci, &chip);

	if (err < 0) {
		printk( "<-----snd_bt87x_probe: Create error ...\n" ) ;
		goto _error;
	}

	if (digital_rate[dev] > 0) {
		printk( "snd_bt87x_probe: digital_rate[%d] > 0 \n", dev ) ;
		chip->dig_rate = digital_rate[dev];
	} else {
		chip->dig_rate = rate ;
	}

	// ST_073000
	err = snd_bt87x_pcm(chip, DEVICE_DIGITAL, "TW6805 Digital");
	// err = snd_bt87x_pcm(chip, DEVICE_ANALOG, "TW68xx Analog");

	if (err < 0) {
		printk( "<-----snd_bt87x_probe: PCM error ...\n" ) ;
		goto _error;
	}

	/*
	err = snd_bt87x_pcm(chip, DEVICE_ANALOG, "TW68xx Analog");

	if (err < 0)	goto _error;
	*/

	err = snd_ctl_add(card, snd_ctl_new1(&snd_bt87x_capture_volume, chip));

	if (err < 0) goto _error;

	// ST_080300, Disable boost mode control
	/*
	err = snd_ctl_add(card, snd_ctl_new1(&snd_bt87x_capture_boost, chip));
	if (err < 0) goto _error;
	*/

	err = snd_ctl_add(card, snd_ctl_new1(&snd_bt87x_capture_source, chip));

	if (err < 0) goto _error;

	strcpy(card->driver, "TW68XX");
	sprintf(card->shortname, "Techwell TW%x", pci->device);
	sprintf(card->longname, "%s at %#llx, irq %i",
		card->shortname, (unsigned long long)pci_resource_start(pci, 0),
		chip->irq);
	strcpy(card->mixername, "TW6805");

	err = snd_card_register(card);
	if (err < 0)
		goto _error;

	pci_set_drvdata(pci, card);
	++dev;
	return 0;

_error:
	snd_card_free(card);
	return err;
}

#else
static int __devinit snd_bt87x_probe(struct pci_dev *pci,
				     const struct pci_device_id *pci_id)
{
	static int dev;
	struct snd_card *card;
	struct snd_bt87x *chip;
	int err;
	enum snd_bt87x_boardid boardid;

	// ST_070900, Add TW68xx Support
	printk( "-----> TW68xx: Probe \n" ) ;

	if (!pci_id->driver_data) {
		err = snd_bt87x_detect_card(pci);
		if (err < 0)
			return -ENODEV;
		boardid = err;
	} else
		boardid = pci_id->driver_data;

	if (dev >= SNDRV_CARDS)
		return -ENODEV;

	if (!enable[dev]) {
		++dev;
		return -ENOENT;
	}

	card = snd_card_new(index[dev], id[dev], THIS_MODULE, 0);

	if (!card)
		return -ENOMEM;

	err = snd_bt87x_create(card, pci, &chip);

	if (err < 0)
		goto _error;

	memcpy(&chip->board, &snd_bt87x_boards[boardid], sizeof(chip->board));

	if (!chip->board.no_digital) {

		if (digital_rate[dev] > 0) chip->board.dig_rate = digital_rate[dev];

		chip->reg_control |= chip->board.digital_fmt;

		// ST_0711
		// err = snd_bt87x_pcm(chip, DEVICE_DIGITAL, "Bt87x Digital");
		err = snd_bt87x_pcm(chip, DEVICE_DIGITAL, "TW68xx Digital");

		if (err < 0) goto _error;
	}

	if (!chip->board.no_analog) {

		// ST_0711
		// err = snd_bt87x_pcm(chip, DEVICE_ANALOG, "Bt87x Analog");
		err = snd_bt87x_pcm(chip, DEVICE_ANALOG, "TW68xx Analog");

		if (err < 0) goto _error;

		err = snd_ctl_add(card, snd_ctl_new1( &snd_bt87x_capture_volume, chip) ) ;

		if (err < 0) goto _error;

		err = snd_ctl_add(card, snd_ctl_new1(  &snd_bt87x_capture_boost, chip) ) ;

		if (err < 0) goto _error;

		err = snd_ctl_add(card, snd_ctl_new1(  &snd_bt87x_capture_source, chip));

		if (err < 0) goto _error;

	}

	snd_printk(KERN_INFO "TW68xx-%d: Using board %d, %sanalog, %sdigital "
		   "(rate %d Hz)\n", dev, boardid,
		   chip->board.no_analog ? "no " : "",
		   chip->board.no_digital ? "no " : "", chip->board.dig_rate);

	// ST_070900, Add TW68xx Support
#if 1
	strcpy(card->driver, "TW68xx");
	sprintf(card->shortname, "Techwell TW%x", pci->device);
	sprintf(card->longname, "%s at %#llx, irq %i",
		card->shortname, (unsigned long long)pci_resource_start(pci, 0),
		chip->irq ) ;
	strcpy(card->mixername, "TW68xx");

#else
	strcpy(card->driver, "Bt87x");
	sprintf(card->shortname, "Brooktree Bt%x", pci->device);
	sprintf(card->longname, "%s at %#llx, irq %i",
		card->shortname, (unsigned long long)pci_resource_start(pci, 0),
		chip->irq);
	strcpy(card->mixername, "Bt87x");
#endif

	err = snd_card_register(card);
	if (err < 0)
		goto _error;

	pci_set_drvdata(pci, card);
	++dev;

	printk( "<----- TW68xx: Probe success ...\n" ) ;

	return 0;

_error:
	snd_card_free(card);

	printk( "<----- TW68xx: Probe exit with error \n" ) ;

	return err;
}

#endif // ST_071400


static void __devexit snd_bt87x_remove(struct pci_dev *pci)
{
	snd_card_free(pci_get_drvdata(pci));
	pci_set_drvdata(pci, NULL);
}

/* default entries for all Bt87x cards - it's not exported */
/* driver_data is set to 0 to call detection */
/*
static struct pci_device_id
snd_bt87x_default_ids[] __devinitdata = {
	// ST_070900, Add TW68xx support
	// BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_878, PCI_ANY_ID, PCI_ANY_ID, UNKNOWN),
	// BT_DEVICE(PCI_DEVICE_ID_BROOKTREE_879, PCI_ANY_ID, PCI_ANY_ID, UNKNOWN),
	BT_DEVICE( PCI_DEVICE_ID_TW6805, PCI_ANY_ID, PCI_ANY_ID, TECHWELL),
	// ST_END
	{ }
};
*/
// ST_070900, Add TW68xx support
#if 1
static struct pci_driver driver = {
	.name = "TW6805",
	.id_table = snd_bt87x_ids,
	.probe = snd_bt87x_probe,
	.remove = __devexit_p(snd_bt87x_remove),
};

#else
static struct pci_driver driver = {
	.name = "Bt87x",
	.id_table = snd_bt87x_ids,
	.probe = snd_bt87x_probe,
	.remove = __devexit_p(snd_bt87x_remove),
};
#endif
// ST_END

static int __init
alsa_card_bt87x_init(void)
{
	// ST_070900_DEL
	/*
	if (load_all)
		driver.id_table = snd_bt87x_default_ids;
	*/

	// ST_070900, Add TW68xx Support
	printk( "===== TW68xx Audio Driver initalize, Ver: 0.01.070900 ===== \n" ) ;

	return pci_register_driver(&driver);
}

static void __exit
alsa_card_bt87x_exit(void)
{

	// ST_070900, Add TW68xx Support
	printk( "===== TW68xx Audio Driver Exit  ===== \n" ) ;

	pci_unregister_driver(&driver);
}

module_init(alsa_card_bt87x_init) ;
module_exit(alsa_card_bt87x_exit) ;


