/*************************************************

videocap.c
used by tw686x

**************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>		/* getopt_long() */
#include <fcntl.h>		/* low-level i/o */
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <asm/types.h>		/* for videodev2.h */
#include <pthread.h>

#include "tw686x.h"
#include "videocap.h"
#include "gui.h"
#include "inifile.h"

/* ioctl for tw686x */
#if(LINUX_VERSION_CODE > KERNEL_VERSION(2,6,30))
#define TW686X_S_REGISTER           _IOWR('V', (BASE_VIDIOC_PRIVATE + 1), struct v4l2_dbg_register)
#define TW686X_G_REGISTER           _IOWR('V', (BASE_VIDIOC_PRIVATE + 2), struct v4l2_dbg_register)
#else
#define TW686X_S_REGISTER           _IOWR('V', (BASE_VIDIOC_PRIVATE + 1), struct v4l2_register)
#define TW686X_G_REGISTER           _IOWR('V', (BASE_VIDIOC_PRIVATE + 2), struct v4l2_register)
#endif
#define TW686X_S_FRAMERATE          _IOWR('V', (BASE_VIDIOC_PRIVATE + 3), long)
#define TW686X_G_FRAMERATE          _IOWR('V', (BASE_VIDIOC_PRIVATE + 4), long)
#define TW686X_G_SIGNAL             _IOWR('V', (BASE_VIDIOC_PRIVATE + 5), long)

static int
xioctl (int fd, int request, void *arg)
{
    int r;

    do
    {
        r = ioctl (fd, request, arg);
    } while (-1 == r && EINTR == errno);

    return r;
}

int
video_device_getframe (channel_t * chl, void **frame_buf, int *frame_len)
{
    struct v4l2_buffer buf;

    fd_set fds;
    struct timeval tv;

    FD_ZERO (&fds);
    FD_SET (chl->dev_video.fd, &fds);

    /* Timeout. */
    tv.tv_sec = 1;
    tv.tv_usec= 0;

    if( select(chl->dev_video.fd + 1, &fds, NULL, NULL, &tv) <= 0) {
        dcprintf (chl, "select time out\r\n");
        return -1;
    }

    pthread_mutex_lock (&(chl->mutex));

    CLEAR (buf);
    buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    buf.memory = V4L2_MEMORY_MMAP;

    if (-1 == xioctl (chl->dev_video.fd, VIDIOC_DQBUF, &buf))
    {
        xioctl (chl->dev_video.fd, VIDIOC_QBUF, &buf);

        pthread_mutex_unlock (&(chl->mutex));

        dcprintf (chl, "VIDIOC_DQBUF failed %d\r\n", chl->error_times);
        chl->error_times++;
        return -1;
    }

    *frame_len = chl->dev_video.buffers[buf.index].length;
    *frame_buf = chl->dev_video.buffers[buf.index].start;

    if( chl->b_show ) {
        memcpy( chl->xvimage->data, *frame_buf, chl->linewidth*chl->height );
        chl->b_update = TRUE;
    }

    if (-1 == xioctl (chl->dev_video.fd, VIDIOC_QBUF, &buf))
    {
        pthread_mutex_unlock (&(chl->mutex));
        dcprintf (chl, "VIDIOC_QBUF failed\r\n");
        return -1;
    }

    pthread_mutex_unlock (&(chl->mutex));

    return 1;
}

int
video_device_stop (channel_t * chl)
{
    enum v4l2_buf_type type;

    type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    return xioctl (chl->dev_video.fd, VIDIOC_STREAMOFF, &type);
}

int
video_device_start (channel_t * chl)
{
    unsigned int i;
    enum v4l2_buf_type type;

    for (i = 0; i < chl->dev_video.n_buffers; ++i)
    {
        struct v4l2_buffer buf;
        CLEAR (buf);
        buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        buf.memory = V4L2_MEMORY_MMAP;
        buf.index = i;

        if (-1 == xioctl (chl->dev_video.fd, VIDIOC_QBUF, &buf))
            return -1;
    }

    type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    return xioctl (chl->dev_video.fd, VIDIOC_STREAMON, &type);
}

int
video_device_mapbuffer (channel_t * chl)
{
    struct v4l2_requestbuffers req;
    char cfg[PATH_MAX];
    char chl_section[PATH_MAX];

    memset(cfg, 0, PATH_MAX);
    get_exe_path(cfg);
    strcat(cfg, CFG_FILE);
    memset(chl_section, 0, PATH_MAX);
    sprintf(chl_section, CFG_SECTION_CHANNEL, chl->id);

    CLEAR (req);

    req.count = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_BUFFERS, MAX_BUF_NUM, cfg);
    req.count = read_profile_int(chl_section, CFG_KEY_BUFFERS, req.count, cfg);
    req.count = min(MAX_BUF_NUM, req.count);

    req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    req.memory = V4L2_MEMORY_MMAP;

    if (-1 == xioctl (chl->dev_video.fd, VIDIOC_REQBUFS, &req))
    {
        return -1;
    }

    if (req.count < 1)
    {
        return -1;
    }

    chl->dev_video.buffers = calloc (req.count, sizeof (*(chl->dev_video.buffers)));

    if (!chl->dev_video.buffers)
    {
        return -1;
    }

    for (chl->dev_video.n_buffers = 0; chl->dev_video.n_buffers < req.count; ++chl->dev_video.n_buffers)
    {
        struct v4l2_buffer buf;

        CLEAR (buf);

        buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        buf.memory = V4L2_MEMORY_MMAP;
        buf.index = chl->dev_video.n_buffers;

        if (-1 == xioctl (chl->dev_video.fd, VIDIOC_QUERYBUF, &buf))
            return -1;

        chl->dev_video.buffers[chl->dev_video.n_buffers].length = buf.length;
        chl->dev_video.buffers[chl->dev_video.n_buffers].start = mmap (NULL /* start anywhere */ ,
                     buf.length,
                     PROT_READ | PROT_WRITE
                     /* required */ ,
                     MAP_SHARED	/* recommended */
                     , chl->dev_video.fd, buf.m.offset);

        if (MAP_FAILED == chl->dev_video.buffers[chl->dev_video.n_buffers].start)
            return -1;
    }
    return 1;
}

int
video_device_init (channel_t * chl)
{
    struct v4l2_capability bcap;

    if (-1 == xioctl (chl->dev_video.fd, VIDIOC_QUERYCAP, &bcap))
    {
        return -1;
    }

    if (!(bcap.capabilities & V4L2_CAP_VIDEO_CAPTURE))
    {
        return -1;
    }

    if (!(bcap.capabilities & V4L2_CAP_STREAMING))
    {
        return -1;
    }

    set_video_standard( chl );
    set_video_resolution( chl );
    set_video_input( chl );

    return video_device_mapbuffer(chl);;
}

int
video_device_close (channel_t * chl)
{
    if (-1 == close (chl->dev_video.fd))
        return -1;
    chl->dev_video.fd = -1;

    return 1;
}

int
video_device_open (channel_t * chl)
{
    struct stat st;

    chl->dev_video.fd = -1;
    memset (chl->dev_video.dev_name, 0, MAX_NAME_LEN);
    sprintf (chl->dev_video.dev_name, "/dev/video%d", chl->id);

    if (-1 == stat (chl->dev_video.dev_name, &st))
    {
        return -1;
    }
    if (!S_ISCHR (st.st_mode))
    {
        return -1;
    }
    chl->dev_video.fd = open (chl->dev_video.dev_name, O_RDWR | O_NONBLOCK /*  required */ , 0);
    if (-1 == chl->dev_video.fd)
    {
        return -1;
    }

    return 1;
}

int
video_device_uninit (channel_t * chl)
{
    unsigned int i;

    for (i = 0; i < chl->dev_video.n_buffers; ++i) {
        if (-1 == munmap (chl->dev_video.buffers[i].start, chl->dev_video.buffers[i].length))
            return -1;
    }

    free (chl->dev_video.buffers);
    return 1;
}

int
get_video_hue (channel_t * chl)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_HUE;
    if (0 == ioctl (chl->dev_video.fd, VIDIOC_G_CTRL, &control))
    {
        res = control.value;
    }

    return res;
}

int
set_video_hue (channel_t * chl, int hue)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_HUE;
    control.value = hue;
    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_CTRL, &control))
    {
        res = -1;
    }

    return res;
}

int
get_video_brightness (channel_t * chl)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_BRIGHTNESS;
    if (0 == ioctl (chl->dev_video.fd, VIDIOC_G_CTRL, &control))
    {
        res = control.value;
    }

    return res;
}

int
set_video_brightness (channel_t * chl, int brightness)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_BRIGHTNESS;
    control.value = brightness;
    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_CTRL, &control))
    {
        res = -1;
    }

    return res;
}

int
get_video_colour (channel_t * chl)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_SATURATION;
    if (0 == ioctl (chl->dev_video.fd, VIDIOC_G_CTRL, &control))
    {
        res = control.value;
    }

    return res;
}

int
set_video_colour (channel_t * chl, int colour)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_SATURATION;
    control.value = colour;
    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_CTRL, &control))
    {
        res = -1;
    }

    return res;
}

int
get_video_contrast (channel_t * chl)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_CONTRAST;
    if (0 == ioctl (chl->dev_video.fd, VIDIOC_G_CTRL, &control))
    {
        res = control.value;
    }

    return res;
}

int
set_video_contrast (channel_t * chl, int contrast)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_CONTRAST;
    control.value = contrast;
    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_CTRL, &control))
    {
        res = -1;
    }

    return res;
}

int
get_video_whiteness (channel_t * chl)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_WHITENESS;
    if (0 == ioctl (chl->dev_video.fd, VIDIOC_G_CTRL, &control))
    {
        res = control.value;
    }

    return res;
}

int
set_video_whiteness (channel_t * chl, int whiteness)
{
    int res = 0;
    struct v4l2_control control;

    memset (&control, 0, sizeof (control));
    control.id = V4L2_CID_WHITENESS;
    control.value = whiteness;
    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_CTRL, &control))
    {
        res = -1;
    }

    return res;
}

int
set_video_standard (channel_t * chl)
{
    v4l2_std_id std_id = chl->standard;

    return ioctl (chl->dev_video.fd, VIDIOC_S_STD, &std_id);
}

int
set_video_resolution (channel_t * chl)
{
    int ret = 0;
    struct v4l2_format fmt;

    CLEAR (fmt);
    fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    fmt.fmt.pix.pixelformat = chl->pix_fmt;

    fmt.fmt.pix.field = chl->interlaced ? V4L2_FIELD_INTERLACED : V4L2_FIELD_TOP;
    fmt.fmt.pix.width = chl->width;
    fmt.fmt.pix.height = chl->height;

    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_FMT, &fmt))
    {
        ret = -1;
        dprintf ("VIDIOC_S_FMT\r\n");
    }

    return ret;
}

int
set_video_framerate (channel_t * chl)
{
    int res = 0;
    long framerate = chl->framerate;

    if (0 != ioctl (chl->dev_video.fd, TW686X_S_FRAMERATE, &framerate))
    {
        res = -1;
        dprintf( "set framerate %d failed\r", (int)framerate ) ;
    }

    return res;
}

int
set_video_input (channel_t * chl)
{
    int res = 0;

    //TODO: change the input number according to the input setting.
    int input = 3; //The Mux0 of TW6805 and TW6816 is input 3

    if (0 != ioctl (chl->dev_video.fd, VIDIOC_S_INPUT, &input))
    {
        res = -1;
        dprintf( "set input to %d failed\r", (int)input ) ;
    }

    return res;
}

int
get_video_framerate (channel_t * chl)
{
    long framerate = 0;

    if (0 != ioctl (chl->dev_video.fd, TW686X_G_FRAMERATE, &framerate))
    {
        framerate = -1;
        dcprintf( chl, "get framerate failed\r\n" ) ;
    }

    return framerate;
}

int
get_video_signal (channel_t * chl)
{
    return 1; //Denio: for TW6816

    long signal = 0;

    if (0 != ioctl (chl->dev_video.fd, TW686X_G_SIGNAL, &signal))
    {
        dcprintf( chl, "get video signal failed\r\n" ) ;
    }

    return signal; 
}

int
set_register ( channel_t * chl, unsigned int addr, unsigned long val )
{
    int ret= 0;
#if(LINUX_VERSION_CODE > KERNEL_VERSION(2,6,30))
    struct v4l2_dbg_register reg;

    memset (&reg, 0, sizeof (reg));
    reg.match.type = V4L2_CHIP_MATCH_HOST;
    reg.size = sizeof( unsigned long );
#else
    struct v4l2_register reg;

    memset (&reg, 0, sizeof (reg));
    reg.match_type = V4L2_CHIP_MATCH_HOST;
#endif
    reg.reg  = addr;
    reg.val  = val;
    ret = ioctl( chl->dev_video.fd, TW686X_S_REGISTER, &reg );

    if( ret != 0 ) {
        dcprintf( chl, "set register 0x%x failed\r\n", addr ) ;
    }

    return ret;
}

unsigned int
get_register ( channel_t * chl, unsigned int addr )
{
    int ret= 0;
#if(LINUX_VERSION_CODE > KERNEL_VERSION(2,6,30))
    struct v4l2_dbg_register reg;

    memset (&reg, 0, sizeof (reg));
    reg.match.type = V4L2_CHIP_MATCH_HOST;
    reg.size = sizeof( unsigned long );
#else
    struct v4l2_register reg;

    memset (&reg, 0, sizeof (reg));
    reg.match_type = V4L2_CHIP_MATCH_HOST;
#endif
    reg.reg  = addr;
    ret = ioctl( chl->dev_video.fd, TW686X_G_REGISTER, &reg );

    if( ret != 0 ) {
        dcprintf( chl, "get register 0x%x failed\r\n", addr ) ;
    }

    return reg.val;
}

int save_registers( channel_t * chl, char* filename )
{
    unsigned int i;
    unsigned int val;
    FILE *f = fopen( filename, "wb" );

    if( f==NULL ) {
        return -EINVAL;
    }

    for( i=0; i<512; i++ ) {
        val = get_register( chl, i<<2 );
        fwrite( &val, sizeof(unsigned char), sizeof(val), f );
    }

    fclose( f );

    return 0;
}
