/*************************************************

tw686x.c
used by tw686x

**************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <linux/kd.h>

#include <getopt.h>		/* getopt_long() */

#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>

#include <asm/types.h>		/* for videodev2.h */

#include <linux/videodev2.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/Xvlib.h>

#include <pthread.h>

#include "tw686x.h"
#include "gui.h"
#include "videocap.h"
#include "audiocap.h"
#include "tw686x-reg.h"
#include "inifile.h"

char * get_exe_path(char *path)
{
    int i;
    int rslt = readlink("/proc/self/exe", path, PATH_MAX);

    if ( rslt < 0 || rslt >= PATH_MAX ) {
        return NULL;
    }
    for(i=rslt-1; i>=0; i--) {
        if(path[i] == '/') {
            i++;
            break;
        }
    }
    path[i] = '\0';
    return path;
}

static int
init_channel (channel_t *chl)
{
    char cfg[PATH_MAX];
    char chl_section[PATH_MAX];
    int  val, pix_depth;

    memset(cfg, 0, PATH_MAX);
    get_exe_path(cfg);
    strcat(cfg, CFG_FILE);
    memset(chl_section, 0, PATH_MAX);
    sprintf(chl_section, CFG_SECTION_CHANNEL, chl->id);

    val = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_STANDARD, 1, cfg);
    val = read_profile_int(chl_section, CFG_KEY_STANDARD, val, cfg);
    chl->standard = (val==0) ? STANDARD_PAL : STANDARD_NTSC;
    val = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_PIXELFORMAT, 0, cfg);
    val = read_profile_int(chl_section, CFG_KEY_PIXELFORMAT, val, cfg);
    switch(val) {
    case 0:
        chl->pix_fmt = V4L2_PIX_FMT_YUYV;
        pix_depth = 16;
        break;
    case 1:
        chl->pix_fmt = V4L2_PIX_FMT_Y41P;
        pix_depth = 12;
        break;
    case 2:
        chl->pix_fmt = V4L2_PIX_FMT_RGB555;
        pix_depth = 16;
        break;
    case 3:
        chl->pix_fmt = V4L2_PIX_FMT_RGB565;
        pix_depth = 16;
        break;
    default:
        chl->pix_fmt = V4L2_PIX_FMT_YUYV;
        break;
    }
    val = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_WIDTH, DEFAULT_VIDEO_WIDTH, cfg);
    val = read_profile_int(chl_section, CFG_KEY_WIDTH, val, cfg);
    chl->width    = min(val, D1_WIDTH);
    val = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_HEIGHT, DEFAULT_VIDEO_HEIGHT(chl->standard), cfg);
    val = read_profile_int(chl_section, CFG_KEY_HEIGHT, val, cfg);
    chl->height   = min(val, D1_HEIGHT(chl->standard));
    chl->linewidth= chl->width * pix_depth / 8;
    chl->dev_video.n_buffers= 0;
    chl->interlaced = (chl->height > CIF_HEIGHT(chl->standard));
    chl->framerate= MAX_FRAMERATE(chl->standard);
    chl->signal   = FALSE;
    chl->b_quit   = FALSE;
    chl->b_run    = FALSE;
    chl->b_show   = TRUE;
    chl->b_update = FALSE;

    dcprintf(chl, "%dx%d \r\n", chl->width,chl->height ) ;

    if (pthread_mutex_init (&(chl->mutex), NULL) != 0)
    {
        return -1;
    }

    if(video_device_init (chl) == -1 )
    {
        dcprintf (chl, "Init video device failed.\r\n");
        return -1;
    }

    chl->audio_bits = 8;
    chl->audio_samplerate = 8000;
    chl->audio_channels = 1;

    return 0;
}

int
capture_init (gui_t * gcap)
{
    int i = 0;
    channel_t *chl;
    char cfg[PATH_MAX];
    int  dev_begin=0, dev_num=0;

    memset(cfg, 0, PATH_MAX);
    get_exe_path(cfg);
    strcat(cfg, CFG_FILE);

    dev_begin = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_DEVICE_BEGIN, 0, cfg);
    dev_num = read_profile_int(CFG_SECTION_TW686X, CFG_KEY_DEVICE_NUM, MAX_CHANNEL_NUM, cfg);
    dev_num = (dev_num==0) ? MAX_CHANNEL_NUM : dev_num;

    dev_num = 4; //denio test

    gcap->formats    = VIDEO_FMT_COUNT;
    gcap->show_curr  = 0;
    gcap->show_begin = 0;
    gcap->show_end   = 0;
    gcap->show_single= FALSE;
    gcap->show_fullscreen = FALSE;

    for(i=0; i<dev_num; i++) {
        chl = &gcap->channel[i];
        chl->id = dev_begin+i;

        if( video_device_open(chl) < 0 ) {
            dcprintf (chl, "Open video device failed.\r\n");
            break;
        }

        if( init_channel(chl) < 0 ) {
            break;
        }

        if( audio_device_open(chl) < 0 ) {
            dcprintf (chl, "Open audio device failed.\r\n");
        }
    }

    gcap->video_count = i;
    gcap->show_end    = gcap->video_count-1;

    return (i-1);
}

int
capture_uninit (void * ctx)
{
    int    i;
    void*  pret = NULL;
    gui_t* gcap = (gui_t*)ctx;
    channel_t *chl;

    gcap->b_quit = TRUE;

    for (i = 0; i < gcap->video_count; i++)
    {
        chl = &gcap->channel[i];
        chl->b_quit = TRUE;
    }

    if( gcap->aud_play_thread!=0 ) {
        pthread_join( gcap->aud_play_thread, &pret );
        gcap->aud_play_thread = 0;
    }

    if( gcap->aud_capture_thread!=0 ) {
        gcap->aud_capture_thread = 0;
    }

    for (i = 0; i < gcap->video_count; i++)
    {
        chl = &gcap->channel[i];

        if( chl->dev_video.cap_thread!=0 ) {
            pthread_join( chl->dev_video.cap_thread, &pret );
        }

        audio_device_close( chl );

        video_device_uninit( chl );
        video_device_close( chl );
    }

    return 0;
}

void *
capture_thread (void *data)
{
    int length = 0;
    channel_t *chl = (channel_t *) data;
    struct timeval tv_begin, tv_now;
    int    framerate = 0, t_count=0;
    long long tk_diff;

    gettimeofday(&tv_begin, NULL);

    chl->error_times = 0;
    chl->signal = get_video_signal(chl);

    //save_registers( chl, "/home/test/work/share/zb.data" );

    while (!chl->b_quit)
    {
        if(chl->signal) {
            if( chl->b_run ) {
                if(video_device_getframe(chl, (void **) &(chl->videobuf), &length) != -1) {
                    framerate++;
                }
            }
            else {
                chl->b_run = (video_device_start(chl) != -1);
                dcprintf (chl, "start %d\r\n", chl->b_run);
            }
        }
        else {
            usleep (10000) ;
            t_count++;
            if(t_count >= 10) {
                chl->b_update = chl->b_show;
                t_count = 0;
            }
        }
        gettimeofday(&tv_now, NULL);
        tk_diff = 1000000 * ( tv_now.tv_sec - tv_begin.tv_sec ) + tv_now.tv_usec - tv_begin.tv_usec;
        tk_diff /= 1000;
        if( tk_diff >= 1000 ) {
            //dcprintf (chl, "framerate=%d %d\r\n", framerate, (int)tk_diff);
            framerate= 0;
            tv_begin = tv_now;
            chl->signal = get_video_signal(chl);
            if( !chl->signal ) {
                if( chl->b_run ) {
                    video_device_stop(chl);
                    chl->b_run = FALSE;
                    dcprintf (chl, "no signal, stop\r\n");
                }
            }
        }
    }

    video_device_stop( chl );

    dcprintf (chl, "video stopped!\r\n");

    return (void *) NULL;
}

void *
audio_capture_thread (void *data)
{
    gui_t *gcap = data;
    channel_t *chl = NULL;
    unsigned char buf[4096];
    int i=0, nread=0;
    wav_file_t wav;

    memset(&wav, 0, sizeof(wav));
    get_exe_path(wav.name);
    strcat(wav.name, "tw686x.wav");
    wav.channels = 1;
    wav.bits     = 8;
    wav.sample_rate = 8000;
    wav.fd = -1;
    wav_open(&wav);

    while(!gcap->b_quit) {
        for(i=0; i<gcap->video_count; i++) {
            chl = &gcap->channel[i];
            if(chl->audio_opened) {
                nread = read(chl->dev_audio.fd, buf, 4096);
                if(nread > 0) {
                    //dcprintf(chl, "read audio bytes %d\n", nread);
                    if((gcap->show_curr == chl->id) && gcap->aud_play_data) {
                        memcpy(gcap->aud_play_data, buf, nread);
                        gcap->aud_play_len = nread;
                        nread = wav_write(&wav, buf, nread);
                    }
                }
            }
        }
        usleep(100);
    }

    wav_close(&wav);

    dprintf ("audio capture stopped!\r\n");

    return (void *) NULL;
}

void *
audio_play_thread (void *data)
{
    gui_t *gcap = data;
    channel_t *chl = NULL;
    unsigned char buf[4096];
    int data_len=0;

    audio_play_open(gcap);
    gcap->aud_play_data = buf;

    while(!gcap->b_quit) {
        chl = &gcap->channel[gcap->show_curr];
        data_len = gcap->aud_play_len;
        if(chl->audio_opened && data_len) {
            gcap->aud_play_len = 0;
            write(gcap->aud_play_fd, buf, data_len);
        }
        else {
            usleep(100);
        }
    }

    audio_play_close(gcap);

    dprintf ("audio play stopped!\r\n");

    return (void *) NULL;
}

int
capture_start( void *ctx )
{
    channel_t *channel;
    gui_t *gcap = ctx;
    int   i;

    for( i=0; i<gcap->video_count; i++ ) {
        channel = &gcap->channel[i];
        if (pthread_create
            (&(channel->dev_video.cap_thread), NULL, capture_thread, channel) != 0)
        {
            dcprintf (channel, "Create capture thread error.\r\n");
            break;
        }
        dcprintf (channel, "capture thread created %d.\r\n", channel->id);
    }

    if (pthread_create(&(gcap->aud_capture_thread), NULL, audio_capture_thread, gcap) != 0) {
        dprintf ("Create audio capture thread error.\r\n");
    }

    if (pthread_create(&(gcap->aud_play_thread), NULL, audio_play_thread, gcap) != 0) {
        dprintf ("Create audio play thread error.\r\n");
    }

    return 0;
}

int
main (int argc, char *argv[])
{
    gui_t gui;

    memset( &gui, 0, sizeof(gui) );
    gui.on_start= capture_start;
    gui.on_quit = capture_uninit;

    if( capture_init(&gui) < 0 )
    {
        dprintf ("Init capture error.\r\n");
        return -1;
    }

    gui_start (&gui);

    dprintf ("TW686X demo exit.\r\n");

    return 0;
}
